/*
 * Ralph.h
 *
 * Created 8/1/2009 By Johnny Huynh
 *
 * Version 00.00.02 8/27/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef RALPH_H
 #define RALPH_H
 
 template <typename T> class Ralph;
 
 #include "CombatCharacter.h"
 
 // For model animations
 #include "auto_bind.h" 
 #include "animControlCollection.h"
 #include "cIntervalManager.h"
 #include "cLerpNodePathInterval.h"
 #include "cMetaInterval.h"
 //#include "genericAsyncTask.h"
 
 // For collision handling
 #include "collisionSegment.h"
 #include "collisionSphere.h"
 #include "collisionParabola.h"
 #include "CollisionHandlerPusherNode.h"
 #include "CollisionHandlerPhysicsNode.h"
 #include "CollisionHandlerEventNodeCollection.h"
 
 #include "global.h"
 #include "StringConversion.h"
 #include "Vector.h"
 #include "DamageObject.h"
 #include "MeleeDamageObject.h"
 
 #include "StraightMovementInfo.h"
 #include "StraightExactMovementInfo.h"
 #include "CurveMovementInfo.h"
 #include "CurveExactMovementInfo.h"
 #include "RotateSegmentMovementInfo.h"
 #include "MeleeAttackInfo.h"
 #include "ProjectileAttackInfo.h"
 #include "SoulOrb.h"
 #include "ProjectileDamageActor.h"
 #include "ActionTask.h"
 
 //#include "StunEffectInfo.h"
 //#include "StraightMoveEffectInfo.h"
 #include "PushBackEffectInfo.h"
 #include "EffectInfoCollection.h"
 
 //NodePath::set_color( 0.184f, 0.725f, 0.725f ); // Miku hair color
 // For safety, remember to undefine these macros at the end of this file
 #define MODEL_DIRECTORY (global::_file_dir + "/models/Miku/Miku_1_4")//"/models/Ralph/ralph")
 #define WALK_ANIMATION_DIRECTORY (global::_file_dir + "/models/Miku/Miku_1_4-anim1")//"/models/Ralph/ralph-walk")
 #define WALK_ANIMATION_NAME "walk"
 #define RUN_ANIMATION_DIRECTORY (global::_file_dir + "/models/Miku/Miku_1_4-anim2")//"/models/Ralph/ralph-run")
 #define RUN_ANIMATION_NAME "run"
 // May want to change name to STD_ATK instead of SLASH
 // and have the DamageObject be binded to a Weapon class,
 // then have the weapon be binded to a character
 //#define DMG_SLASH_NAME "slash"
 #define DMG_SIDE_SLASH_NAME "side_slash"
 #define DMG_DOWN_SLASH_NAME "down_slash"
 
 #define MODEL_SCALE 1.0f
 
 // Movement speeds measured in units
 #define WALKING_SPEED 0.3f
 #define RUNNING_SPEED 0.5f
 
 /**
  * Class specification for Ralph
  */
 template <typename T>
 class Ralph : public CombatCharacter<T>
 {
 // Data Members
 private:
    unsigned int _std_atk_combo; // should probably be moved to Ralph or branch into Character Class
    double _time_of_last_standard_attack;
    // ActionType _previous_action; (enum)
 
 // Static Functions
 public:
    //static inline void execute_test( const Event * event_Ptr, void * data_Ptr );
    static inline void dash( const Event * event_Ptr, void * data_Ptr );
    static inline void execute_projectile_attack( const Event * event_Ptr, void * data_Ptr );
    static inline void execute_standard_attack( const Event * event_Ptr, void * data_Ptr );
 
 // Local Functions
 public:
    Ralph();
    Ralph( const Ralph<T>& ralph );
    virtual ~Ralph();
    inline Ralph<T>& operator=( const Ralph<T>& ralph );
    virtual inline void assign_interval( WindowFramework& window );
    virtual inline std::string get_default_collider_name() const;
    virtual inline void load_model( WindowFramework& window );
    virtual inline void load_model( WindowFramework& window, const NodePath& parent );
    virtual inline void run_forward();
    virtual inline void assume_neutral_position();
    virtual inline void walk_forward();
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "Ralph<" + template_type + ">" );
                            }*/
 
 // Private Functions
 private:
    virtual inline void initialize_collision_handler();
    virtual inline void initialize_damage_objects( WindowFramework& window );
    
 // Friend Functions
 public:
    
 };
 
 /** STATIC FUNCTIONS **/
 
 /*template <typename T>
 inline void Ralph<T>::execute_test( const Event * event_Ptr, void * data_Ptr )
 {
    PT(Ralph<T>) ralph_Ptr( static_cast<Ralph<T>*>( data_Ptr ) );
    for ( size_t i(0); i < 9; ++i )
    {
        ralph_Ptr->execute_projectile_attack( event_Ptr, data_Ptr );
        //if ( global::_task_mgr_Ptr->get_num_tasks() > 0 )
            //global::_task_mgr_Ptr->remove( global::_task_mgr_Ptr->get_tasks() );
        //if ( global::_action_task_mgr_Ptr->get_num_tasks() > 0 )
            //global::_action_task_mgr_Ptr->remove( global::_action_task_mgr_Ptr->get_tasks() );
    }
 }*/
 
 /**
  * dash() makes the Ralph pointed to (as user data) by the specified GenericAsyncTask perform a dash.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void Ralph<T>::dash( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    
    PT(Ralph<T>) ralph_Ptr( static_cast<Ralph<T>*>( data_Ptr ) );
    
    // if the character is stunned, the character cannot perform the action
    if ( ralph_Ptr->is_stunned() )
        return;
    
    // clear animations and colliders
    // change the Object's animation and collider
    ralph_Ptr->CollisionActor<T>::pose( RUN_ANIMATION_NAME, 4 );
    
    MovementInfoCollection<T> movement_info_collection;
    
    VECTOR2_TYPE dir_vec2( Vector::get_xy_direction_normalized( ralph_Ptr->get_h() ) );
    
    VECTOR3_TYPE forward_vector( dir_vec2.get_x(), dir_vec2.get_y(), ZERO );
    //VECTOR3_TYPE up_vector( ZERO, ZERO, ONE );
    //VECTOR3_TYPE left_vector( up_vector.cross( forward_vector ) );
    //VECTOR3_TYPE acceleration( left_vector );
    //VECTOR3_TYPE velocity( -left_vector + forward_vector );
    PT(StraightMovementInfo<T>) movement_info_Ptr_01( new StraightMovementInfo<T>( forward_vector, 0.0f, 0.03 ) );
    PT(StraightMovementInfo<T>) movement_info_Ptr_02( new StraightMovementInfo<T>( forward_vector, 8.5f, 0.09 ) );
    //PT(CurveExactMovementInfo<T>) movement_info_Ptr( new CurveExactMovementInfo<T>( acceleration, velocity, ralph_Ptr->get_pos() + (2*(forward_vector + up_vector)), 55.0f, 1.0 ) );
    movement_info_collection.add( movement_info_Ptr_01 );
    movement_info_collection.add( movement_info_Ptr_02 );
    
    PT(PushInfo<T>) push_info_Ptr( new PushInfo<T>( 0.25, movement_info_collection  ) );
    PT(ActionTask<T>) action_task_Ptr( new ActionTask<T>( ralph_Ptr, push_info_Ptr ) );
    action_task_Ptr->add_to_task_manager();
 }
 
 /**
  * execute_projectile_attack() makes the Ralph pointed to (as user data) 
  * by the specified GenericAsyncTask perform a projectile attack.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void Ralph<T>::execute_projectile_attack( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    //printf( "Ralph 120 - RefCount: %1d\n", static_cast<Ralph<T>*>( data_Ptr )->get_ref_count() );
    PT(Ralph<T>) ralph_Ptr( static_cast<Ralph<T>*>( data_Ptr ) );
    //printf( "Ralph 122 - RefCount: %1d\n", static_cast<Ralph<T>*>( data_Ptr )->get_ref_count() );
    
    // if the character is stunned, the character cannot attack
    if ( ralph_Ptr->is_stunned() )
        return;
    
    // clear animations and colliders
    // change the Object's animation and collider
    ralph_Ptr->CollisionActor<T>::pose( RUN_ANIMATION_NAME, 1 );
    
    MovementInfoCollection<T> movement_info_collection;
    
    VECTOR2_TYPE dir_vec2( Vector::get_xy_direction_normalized( ralph_Ptr->get_h() ) );
    
    VECTOR3_TYPE forward_vector( dir_vec2.get_x(), dir_vec2.get_y(), ZERO );
    VECTOR3_TYPE up_vector( ZERO, ZERO, ONE );
    VECTOR3_TYPE left_vector( up_vector.cross( forward_vector ) );
    VECTOR3_TYPE acceleration( left_vector );
    VECTOR3_TYPE velocity( -left_vector + forward_vector );
    //PT(StraightMovementInfo<T>) movement_info_Ptr( new StraightMovementInfo<T>( forward_vector, 20.0f, 1.0 ) );
    PT(CurveExactMovementInfo<T>) movement_info_Ptr( new CurveExactMovementInfo<T>( acceleration, velocity, ralph_Ptr->get_pos() + (2*(forward_vector + up_vector)), 55.0f, 1.0 ) );
    movement_info_collection.add( movement_info_Ptr );
    
    PT(SoulOrb<T>) soul_orb_Ptr( new SoulOrb<T>( "Ralph's Soul Orb" ) );
    // Position the SoulOrb
    soul_orb_Ptr->set_pos( ralph_Ptr->get_pos() + ((2*forward_vector) + (2*up_vector)) ); // if have gravity, then move towards ground position
    
    PT(ProjectileAttackInfo<T>) proj_attack_info_Ptr( new ProjectileAttackInfo<T>( ralph_Ptr->get_parent(), soul_orb_Ptr, 0.0, 1.0, movement_info_collection ) );
    PT(ActionTask<T>) action_task_Ptr( new ActionTask<T>( ralph_Ptr, proj_attack_info_Ptr ) );
    action_task_Ptr->add_to_task_manager();
    
    // stun the character
    PT(StunInfo<T>) stun_info_Ptr( new StunInfo<T>( 0.15 ) );
    PT(ActionTask<T>) ralph_action_task_Ptr( new ActionTask<T>( ralph_Ptr, stun_info_Ptr ) );
    ralph_action_task_Ptr->add_to_task_manager();
 }
 
 /**
  * execute_standard_attack() makes the Ralph pointed to (as user data) 
  * by the specified GenericAsyncTask perform a standard attack.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 template <typename T>
 inline void Ralph<T>::execute_standard_attack( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    //printf( "Ralph 120 - RefCount: %1d\n", static_cast<Ralph<T>*>( data_Ptr )->get_ref_count() );
    PT(Ralph<T>) ralph_Ptr( static_cast<Ralph<T>*>( data_Ptr ) );
    //printf( "Ralph 122 - RefCount: %1d\n", static_cast<Ralph<T>*>( data_Ptr )->get_ref_count() );
    
    // if the character is stunned, the character cannot attack
    if ( ralph_Ptr->is_stunned() )
        return;
    
    // add attack collider
    PT(DamageObject<T>) dmg_slash_object_Ptr;//( ralph_Ptr->get_damage_object( DMG_SLASH_01_NAME ) );
    //nassertv( dmg_slash_object_Ptr != NULL );
    // attach appriopriate slash collider to dmg_slash_obj
    //ColliderCollection& slash_colliders( dmg_slash_object_Ptr->get_colliders() );
    //dmg_object_Ptr->turn_off_collision();
    //slash_colliders.detach_colliders();
    //dmg_slash_object_Ptr->detach_colliders();
    
    // attach dmg_slash_obj to parent of Ralph
    //NodePath parent(ralph_Ptr->get_parent());
    
    //nassertv( !parent.is_empty() );
    
    //dmg_slash_object_Ptr->set_pos( ralph_Ptr->get_pos() );
    //dmg_slash_object_Ptr->set_hpr( ralph_Ptr->get_hpr() );
    //dmg_slash_object_Ptr->reparent_to( *ralph_Ptr );
    
    // Combo if within time of consecutive attacks
    double current_time( global::_clock_Ptr->get_real_time() );
    if ( current_time - ralph_Ptr->_time_of_last_standard_attack > 0.75 || ralph_Ptr->_std_atk_combo == 0 )
    {
        // clear animations and colliders
        // change the Object's animation and collider
        ralph_Ptr->CollisionActor<T>::pose( RUN_ANIMATION_NAME, 3 );
        
        // change the Object's damage collider
        //dmg_slash_object_Ptr->CollisionObject<T>::reparent_collider_to( DMG_SLASH_01_NAME, *dmg_slash_object_Ptr ); // may want function, attach_collider()
        dmg_slash_object_Ptr = ralph_Ptr->get_damage_object( DMG_SIDE_SLASH_NAME );
        nassertv( dmg_slash_object_Ptr != NULL );
        
        ++ralph_Ptr->_std_atk_combo; // increment standard attack combo
    }
    else // current_time - ralph_Ptr->_time_of_last_standard_attack <= 0.75
    {
        // clear animations and colliders
        // change the Object's animation and collider
        ralph_Ptr->CollisionActor<T>::pose( RUN_ANIMATION_NAME, 6 );
        
        // change the Object's damage collider
        //dmg_slash_object_Ptr->CollisionObject<T>::reparent_collider_to( DMG_SLASH_02_NAME, *dmg_slash_object_Ptr );
        dmg_slash_object_Ptr = ralph_Ptr->get_damage_object( DMG_DOWN_SLASH_NAME );
        nassertv( dmg_slash_object_Ptr != NULL );
        
        ralph_Ptr->_std_atk_combo = 0; // reset standard attack combo
    }
    
    ralph_Ptr->_time_of_last_standard_attack = global::_clock_Ptr->get_real_time(); // set time of current standard attack
    // call task manager function to change the value of _movement_speed after a specified amount of time
    // and set this object to assume its neutral stance
    // and turn off from collision for the collider (e.g. slash_01_Ptr)
    //ralph_Ptr->add_task( Ralph<T>::reset_standard_attack, static_cast<void*>(ralph_Ptr) );
    
    MovementInfoCollection<T> movement_info_collection;
    
    VECTOR2_TYPE dir_vec2( Vector::get_xy_direction_normalized( ralph_Ptr->get_h() ) );
    VECTOR3_TYPE dir1( dir_vec2.get_x(), dir_vec2.get_y(), ZERO );
    VECTOR3_TYPE dir2( ZERO, ZERO, ZERO );
    VECTOR3_TYPE dir3( dir_vec2.get_x(), dir_vec2.get_y(), ZERO );
    //PT(StraightExactMovementInfo<T>) movement_info_Ptr_1( new StraightExactMovementInfo<T>( ralph_Ptr->get_pos(), dir1, 2.0f, 0.15 ) );
    VECTOR3_TYPE forward_vector( dir_vec2.get_x(), dir_vec2.get_y(), ZERO );
    VECTOR3_TYPE up_vector( ZERO, ZERO, ONE );
    VECTOR3_TYPE left_vector( up_vector.cross( forward_vector ) );
    VECTOR3_TYPE acceleration( left_vector );
    VECTOR3_TYPE velocity( -left_vector + forward_vector );
    //PT(CurveMovementInfo<T>) movement_info_Ptr_1( new CurveMovementInfo<T>( acceleration, velocity, 15.0f, 0.15 ) );
    //PT(CurveExactMovementInfo<T>) movement_info_Ptr_1( new CurveExactMovementInfo<T>( acceleration, velocity, ralph_Ptr->get_pos(), 15.0f, 0.15 ) );
    PT(StraightMovementInfo<T>) movement_info_Ptr_1( new StraightMovementInfo<T>( dir1, 4.0f, 0.1 ) );
    //PT(StraightMovementInfo<T>) movement_info_Ptr_2( new StraightMovementInfo<T>( dir2, ZERO, 0.5 ) );
    //PT(StraightMovementInfo<T>) movement_info_Ptr_3( new StraightMovementInfo<T>( dir3, -5.0f, 0.25 ) );
    movement_info_collection.add( movement_info_Ptr_1 );
    //movement_info_collection.add( movement_info_Ptr_2 );
    //movement_info_collection.add( movement_info_Ptr_3 );
    
    PT(MeleeAttackInfo<T>) melee_attack_info_Ptr( new MeleeAttackInfo<T>( dmg_slash_object_Ptr, 0.1, 0.3, 0.3, movement_info_collection ) );
    //PT(MeleeAttackInfo<T>) melee_attack_info_Ptr( new MeleeAttackInfo<T>( dmg_slash_object_Ptr, 0.0, 0.2, 0.2, movement_info_collection ) );
    PT(ActionTask<T>) action_task_Ptr( new ActionTask<T>( ralph_Ptr, melee_attack_info_Ptr ) );
    action_task_Ptr->add_to_task_manager(); //global::_action_task_mgr_Ptr );
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 Ralph<T>::Ralph()
          : CombatCharacter<T>( RUNNING_SPEED ),
            _std_atk_combo( 0 ),
            _time_of_last_standard_attack( 0.0 )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 Ralph<T>::Ralph( const Ralph<T>& ralph )
          : CombatCharacter<T>( static_cast< CombatCharacter<T> >( ralph ) ),
            _std_atk_combo( ralph._std_atk_combo ),
            _time_of_last_standard_attack( 0.0 )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 Ralph<T>::~Ralph()
 {
    //printf( "ref count: %1d\n", _collider_Ptr->get_ref_count() );
    //_collider_Ptr->remove_node();
    
    //printf( "Number of Children: %1d\n", _collider_Ptr->get_num_children() );
    //printf( "Number of Parents: %1d\n", _collider_Ptr->get_num_parents() );
    //printf( "ref count: %1d\n", _collider_Ptr->get_ref_count() );
    //printf( ("Parent Name: " + _collider_Ptr->get_parent(0)->get_name() + "\n").c_str() );
 }
 
 /**
  * operator=() copies the content of the specified Ralph to this Ralph.
  *
  * @param (const Ralph<T>&) ralph
  * @return Ralph<T>&
  */
 template <typename T>
 inline Ralph<T>& Ralph<T>::operator=( const Ralph<T>& ralph )
 {
    CombatCharacter<T>::operator=( static_cast< CombatCharacter<T> >( ralph ) );
    _std_atk_combo = ralph._std_atk_combo;
    _time_of_last_standard_attack = 0.0;
    
    return *this;
 }
 
 /**
  * assign_interval() assigns an interval (movement instructions) to
  * Ralph.
  */
 template <typename T>
 inline void Ralph<T>::assign_interval( WindowFramework& window )
 {
    // Create the lerp intervals needed to walk back and forth
    PT(CLerpNodePathInterval) ralphPosInterval1, ralphPosInterval2, ralphHprInterval1, ralphHprInterval2;
    ralphPosInterval1 = new CLerpNodePathInterval("ralphPosInterval1", 13.0, CLerpInterval::BT_no_blend,
                                                      true, false, static_cast< NodePath >( *this ), window.get_render());
    if ( ralphPosInterval1 != NULL )
    {
        ralphPosInterval1->set_start_pos(LPoint3f(0, 10, 0) + get_pos());
        ralphPosInterval1->set_end_pos(LPoint3f(0, -10, 0) + get_pos());
    }

    ralphPosInterval2 = new CLerpNodePathInterval("ralphPosInterval2", 13.0, CLerpInterval::BT_no_blend,
                                                      true, false, static_cast< NodePath >( *this ), window.get_render());
    if ( ralphPosInterval2 != NULL )
    {
        ralphPosInterval2->set_start_pos(LPoint3f(0, -10, 0) + get_pos());
        ralphPosInterval2->set_end_pos(LPoint3f(0, 10, 0) + get_pos());
    }

    ralphHprInterval1 = new CLerpNodePathInterval("ralphHprInterval1", 3.0, CLerpInterval::BT_no_blend,
                                                      true, false, static_cast< NodePath >( *this ), window.get_render());
    if ( ralphHprInterval1 != NULL )
    {
        ralphHprInterval1->set_start_hpr(LPoint3f(180.0f, ZERO, ZERO) + get_hpr());
        ralphHprInterval1->set_end_hpr(LPoint3f(ZERO, ZERO, ZERO) + get_hpr());
    }

    ralphHprInterval2 = new CLerpNodePathInterval("ralphHprInterval2", 3.0, CLerpInterval::BT_no_blend,
                                                      true, false, static_cast< NodePath >( *this ), window.get_render());
    if ( ralphHprInterval2 != NULL )
    {
        ralphHprInterval2->set_start_hpr(LPoint3f(ZERO, ZERO, ZERO) + get_hpr());
        ralphHprInterval2->set_end_hpr(LPoint3f(180.0f, ZERO, ZERO) + get_hpr());
    }

    // Create and play the sequence that coordinates the intervals
    PT(CMetaInterval) ralphPace( new CMetaInterval("ralphPace") );
    if ( ralphPace != NULL )
    {
        // The following uses sequential animation. For parallel animation, use
        // CMetaInterval::RS_previous_begin instead.
        ralphPace->add_c_interval(ralphHprInterval1, 0.0, CMetaInterval::RS_previous_end);
        ralphPace->add_c_interval(ralphPosInterval2, 0.0, CMetaInterval::RS_previous_begin);//CMetaInterval::RS_previous_end);
        ralphPace->add_c_interval(ralphHprInterval2, 0.0, CMetaInterval::RS_previous_end);
        ralphPace->add_c_interval(ralphPosInterval1, 0.0, CMetaInterval::RS_previous_begin);//end);
        ralphPace->loop();
    }
    /*int index( CIntervalManager::get_global_ptr()->find_c_interval( "ralphPace" ) );
    printf( "%1d\n", index );
    CIntervalManager::get_global_ptr()->remove_c_interval( index );
    printf( "%1d\n", CIntervalManager::get_global_ptr()->get_num_intervals() );*/
    
    //global::_task_mgr_Ptr->add( new GenericAsyncTask( "Process Interval Events", processIntervals, (void*) NULL ) );
 }
 
 /**
  * get_default_collision_name() returns the name of the default collider for this Object.
  *
  * @return std::string
  */
 template <typename T>
 inline std::string Ralph<T>::get_default_collider_name() const
 {
    return WALK_ANIMATION_NAME;
 }
 
 /**
  * load_model() loads the Object model for the specified WindowFramework. For the
  * Object to be visible in the world, it needs to be reparented to a NodePath
  * that is part of the specified WindowFramework's scene graph.
  *
  * @param (WindowFramework&) window
  */
 template <typename T> 
 inline void Ralph<T>::load_model( WindowFramework& window )
 {
    Object<T>::load_model( window, MODEL_DIRECTORY );
 }
 
 /**
  * load_model() loads the Object model into the specified WindowFramework under
  * the specified parent in the scene graph of the specified WindowFramework.
  *
  * @param (WindowFramework&) window
  * @param (const NodePath&) parent
  */
 template <typename T> 
 inline void Ralph<T>::load_model( WindowFramework& window, const NodePath& parent )
 {
    Object<T>::load_model( window, MODEL_DIRECTORY, parent );
    
    // If unable to load the model, then exit the function
    if ( NodePath::is_empty() )
        return;
    
    NodePath::set_scale( MODEL_SCALE );
    
    // Load the walk animation
    NodePath walkAnimation( window.load_model( static_cast< NodePath >( *this ), WALK_ANIMATION_DIRECTORY ) );
    
    // If unable to load the walking animation, then exit the function
    if ( walkAnimation.is_empty() )
        return;
    // Load the run animation
    NodePath runAnimation( window.load_model( static_cast< NodePath >( *this ), RUN_ANIMATION_DIRECTORY ) );
    
    // If unable to load the running animation, then exit the function
    if ( runAnimation.is_empty() )
        return;
    //printf( "%1d\n", node_path.node()->get_num_children() );
    //node_path.node()->remove_child( walkAnimation.node() );
    // replace_child
    //printf( "%1d\n", node_path.node()->get_num_children() );
    
    //node_path.node()->remove_child(0);
    
    AnimControlCollection& anim_collection( Actor<T>::get_anims() );
    
    auto_bind( NodePath::node(), anim_collection, PartGroup::HMF_ok_part_extra | PartGroup::HMF_ok_anim_extra | PartGroup::HMF_ok_wrong_root_name ); // equivalent to 0x7
    
    // We bind the walk and run animations to anim_collection, and then remove 
    // the animations and store them in anim_collection again so that we can
    // manually assign custom keys to the animations for later retrieval; unfortunely,
    // there is no other way to change the key of an AnimControl inside of 
    // an AnimControlCollection.
    PT(AnimControl) walk_anim( anim_collection.get_anim(1) );
    PT(AnimControl) run_anim( anim_collection.get_anim(0) );
    anim_collection.clear_anims();
    anim_collection.store_anim( walk_anim, WALK_ANIMATION_NAME );
    anim_collection.store_anim( run_anim, RUN_ANIMATION_NAME );
    
    // Reorientation of the Ralph model because the model is facing 180 degrees 
    // in the wrong direction of the heading angle
    // Top: node_path
    // 2nd: *this (<-- contains all the model and animations)
    // Set *this = node_path (so that the top node is controlled, leaving the 2nd 
    // NodePath perpetually rotated by 180 degrees)
    // CAVEAT: You should not need this block of code if your 3D model is oriented properly
    PT(PandaNode) panda_node( new PandaNode( "Ralph Model Reorientation" ) );
    NodePath node_path( NodePath::get_parent().attach_new_node( panda_node ) );
    NodePath::set_h( 180.0f );
    //NodePath::set_transform( NodePath::get_transform()->make_hpr( LVecBase3f( 180.0f, ZERO, ZERO ) ) );
    NodePath::reparent_to( node_path );
    NodePath::operator=( node_path );
    
    // The names of the anim seem meaningless
    // The strings assigned to the animations (for store_anim) are the names that matters
    /*for ( size_t i( 0 ); i < anim_collection.get_num_anims(); ++i )
    {
        printf( "%1d\n", anim_collection.get_num_anims() );
        printf( anim_collection.get_anim( i )->get_name().c_str() );
        printf("\n");
    }*/
    /*anim_collection.get_anim(0)->clear_name();
    anim_collection.get_anim(0)->set_name( "ralph-walk" );*/
    //window.loop_animations( 0 );
    //anim_collection.loop( WALK_ANIMATION_NAME, true );
    //printf( "%1d\n", node_path.node()->get_num_children() );
    //panda_anims.loop_all( true );
    //printf( panda_anims.get_anim_name( 0 ).c_str() );
    
    // Apply collision handling
    // This should occur after the animations are loaded because collision only effect a collider's parent,
    // and the parent should be this NodePath
    initialize_collision_handler();
    
    assume_neutral_position();
    
    initialize_damage_objects( window );
 }
 
 /**
  * run_forward() makes Ralph runs forward.
  */
 template <typename T>
 inline void Ralph<T>::run_forward()
 {
    CollisionActor<T>::loop( RUN_ANIMATION_NAME );
    
    if ( CharacterObject<T>::get_movement_speed() != RUNNING_SPEED )
        CharacterObject<T>::set_movement_speed( RUNNING_SPEED );
 }
 
 /**
  * assume_neutral_position() positions the Actor into the Actor's neutral stance.
  */
 template <typename T>
 inline void Ralph<T>::assume_neutral_position()
 {
    CollisionActor<T>::pose( WALK_ANIMATION_NAME, 5 );
 }
 
 /**
  * walk_forward() makes Ralph walks forward.
  */
 template <typename T>
 inline void Ralph<T>::walk_forward()
 {
    CollisionActor<T>::loop( WALK_ANIMATION_NAME );
    
    if ( CharacterObject<T>::get_movement_speed() != WALKING_SPEED )
        CharacterObject<T>::set_movement_speed( WALKING_SPEED );
 }
 
 /** PRIVATE FUNCTIONS **/
 
 /**
  * initialize_collision_handler() initializes the collision handler for this object.
  */
 template <typename T>
 inline void Ralph<T>::initialize_collision_handler()
 {
    // During each frame (when the game runs its course), each Object should have no more than one CollisionHandlerPusherNode
    // or CollisionHandlerPhysicsNode (i.e. only one or zero collider for an Object should be active)
    // Upon initialization of a collider, the collider will automatically be set to active. Use detach_node on the collider
    // to deactivate the collider. Similarly, use the reparent_to function of the collider to activate a collider.
    
    //ColliderCollection& colliders( CollisionObject<T>::get_colliders() );
    
    //std::string object_key( StringConversion::to_str( NodePath::get_key() ) );
    
    PT(CollisionHandlerPusherNode<T>) walk_collider_Ptr( new CollisionHandlerPusherNode<T>( WALK_ANIMATION_NAME ) );
    //walk_collider_Ptr->set_tag( "Object Key", object_key );
    //colliders.add( WALK_ANIMATION_NAME, walk_collider_Ptr );
    CollisionObject<T>::add_collider( walk_collider_Ptr );
    
    walk_collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, ZERO, 1.0f, 1.0f ) );
    walk_collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, ZERO, 2.5f, 1.0f ) ); // 2.5f
    walk_collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, ZERO, 4.2f, 1.0f ) );
    //walk_collider_Ptr->reparent_to( *this );
    //walk_collider_Ptr->initialize_collision_handler( *this );
    //walk_collider_Ptr->turn_on_from_collision();
    walk_collider_Ptr->show(); // Show the collision solid
    
    // Event Handling
    walk_collider_Ptr->add_in_pattern(); // generate an in-pattern collision
    walk_collider_Ptr->add_event(); // handle the in-pattern collision
    
    PT(CollisionHandlerPusherNode<T>) run_collider_Ptr( new CollisionHandlerPusherNode<T>( RUN_ANIMATION_NAME ) );
    //run_collider_Ptr->set_tag( "Object Key", object_key );
    //colliders.add( RUN_ANIMATION_NAME, run_collider_Ptr );
    CollisionObject<T>::add_collider( run_collider_Ptr );
    
    run_collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, ZERO, 1.0f, 1.0f ) );
    run_collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, ZERO, 2.5f, 1.0f ) ); // 2.5f
    run_collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, 0.75f, 4.2f, 1.0f ) );
    //run_collider_Ptr->reparent_to( *this );
    //run_collider_Ptr->initialize_collision_handler( *this );
    //run_collider_Ptr->turn_on_from_collision();
    run_collider_Ptr->show(); // Show the collision solid
    //run_collider_Ptr->detach_node();
    //walk_collider_Ptr->reparent_to( *this );
    
    // Event Handling
    //printf( (object_key + "\n").c_str() );
    //printf( ("Tag: " + run_collider_Ptr->get_net_tag( "Object Key" ) + "\n").c_str() );
    run_collider_Ptr->add_in_pattern(); // generate an in-pattern collision
    run_collider_Ptr->add_event(); // handle the in-pattern collision
    //run_collider_Ptr->add_in_pattern( object_key + "-into"  );//%in" );
    //printf( (_collider_Ptr->get_in_pattern( 0 ) + "\n").c_str() );
    //printf( _collider_Ptr->get_name().c_str() );
    //printf("\n");
    
    //global::_framework.get_event_handler().add_hook( object_key + "-into", &handle_collision<T> );//_collider_Ptr->get_name() + "%(" + object_key + ")fh", &testFunction );
    
    // Should have a way to remove the hooks
    // DONE! (Handled by the CollisionHandlerEventNode upon calling destructor)
    //global::_framework.get_event_handler().remove_hooks( object_key + "-into" );
 }
 
 /**
  * initialize_damage_objects() initializes the damage objects belonging to this object.
  *
  * @param (WindowFramework&) window
  */
 template <typename T>
 inline void Ralph<T>::initialize_damage_objects( WindowFramework& window )
 {
    std::string object_key( StringConversion::to_str( NodePath::get_key() ) );
    // CollisionLine
    // Slash Attacks:
    {
        //PT(StunEffectInfo<T>) stun_effect_info_Ptr( new StunEffectInfo<T>( 0.15 ) );
        //PT(StraightMoveEffectInfo<T>) straight_move_effect_info_Ptr( new StraightMoveEffectInfo<T>( 1.5f, 0.15 ) );
        EffectInfoCollection<T> effect_info_collection;
        //effect_info_collection.add( stun_effect_info_Ptr );
        //effect_info_collection.add( straight_move_effect_info_Ptr );
        PT(PushBackEffectInfo<T>) push_back_effect_info_Ptr( new PushBackEffectInfo<T>( 0.15, 1.5f, 0.15 ) );
        effect_info_collection.add( push_back_effect_info_Ptr );
        
        // P(t) = at^2 + vt + p
        T distance( 2.2f );//1.7f );
        LVecBase3f ep_a( -6.5f, -9.0f, ZERO ); //-5.5f, -9.0f, ZERO );
        LVecBase3f ep_v( ZERO, 9.0f, 0.6f ); //ZERO, 9.0f, 1.0f );
        LVecBase3f ep_p( 3.5f, 0.5f, 1.5f ); //2.5f, 3.0f, 2.0f );
        LVecBase3f origin( ZERO, ONE, 2.0f );
        PT(CollisionSegment) collision_segment_Ptr( new CollisionSegment( origin, ep_p ) );
        
        //PT(DamageObject<T>) slash_Ptr( new DamageObject<T>( DMG_SLASH_NAME, 5.0f, effect_info_collection ) );
        PT(DamageObject<T>) side_slash_Ptr( new MeleeDamageObject<T>( DMG_SIDE_SLASH_NAME, 5.0f, effect_info_collection,
                                                                 collision_segment_Ptr, ep_a, ep_v, distance*ep_p, distance, 0.09 ) );
        CombatCharacter<T>::add_damage_object( side_slash_Ptr );
        
        PT(CollisionHandlerEventNode<T>) side_slash_collider_Ptr( new CollisionHandlerEventNode<T>( DMG_SIDE_SLASH_NAME ) );
        side_slash_Ptr->DamageObject<T>::add_collider( side_slash_collider_Ptr );
        side_slash_collider_Ptr->collision_node()->add_solid( collision_segment_Ptr );
        
        side_slash_collider_Ptr->show();
        side_slash_collider_Ptr->reparent_to( *side_slash_Ptr );
        
        //side_slash_collider_Ptr->turn_on_from_collision();
        //side_slash_collider_Ptr->detach_node();
        
        // Event Handling
        side_slash_collider_Ptr->add_in_pattern();
        side_slash_collider_Ptr->add_event();
        
        // 2) Down-Slash (left to right)
        /*PT(CollisionHandlerEventNode<T>) down_slash_collider_Ptr( new CollisionHandlerEventNode<T>( DMG_SLASH_02_NAME ) );
        slash_Ptr->DamageObject<T>::add_collider( down_slash_collider_Ptr );
        // P(t) = at^2 + vt + p
        ep_a = LVecBase3f( 2.0f, -4.0f, -4.0f ); //2.0f, -3.5f, -3.0f ); // acceleration
        ep_v = LVecBase3f( ZERO, 5.5f, -1.0f ); //ZERO, 4.5f, -1.0f ); // velocity
        ep_p = LVecBase3f( -0.75f, 2.5f, 6.0f ); //-0.75f, 2.5f, 5.0f ); // begin point
        down_slash_collider_Ptr->collision_node()->add_solid( new CollisionParabola( Parabolaf( ep_a, ep_v, ep_p ), ZERO, ONE  ) ); // Parabolaf(a, b, c), t1, t2
        //down_slash_collider_Ptr->reparent_to( *slash_Ptr );
        down_slash_collider_Ptr->show();
        
        //down_slash_collider_Ptr->turn_on_from_collision();
        //down_slash_collider_Ptr->detach_node();
        
        // Event Handling
        down_slash_collider_Ptr->add_in_pattern();
        down_slash_collider_Ptr->add_event();*/
        
        //_dmg_objects.get( DMG_SLASH_NAME )->detach_node();
        //slash_Ptr->detach_node();
        // P(t) = at^2 + vt + p
        
        distance = 1.7f; //2.0f;
        ep_a = LVecBase3f( 2.0f, -4.0f, -4.0f ); //2.0f, -3.5f, -3.0f ); // acceleration
        ep_v = LVecBase3f( ZERO, 5.5f, -1.0f ); //ZERO, 4.5f, -1.0f ); // velocity
        ep_p = LVecBase3f( -0.75f, 2.5f, 6.0f ); //-0.75f, 2.5f, 5.0f ); // begin point
        //origin = LVecBase3f( ZERO, ONE, 2.0f );
        collision_segment_Ptr = PT(CollisionSegment)(new CollisionSegment( origin, ep_p ));
        
        //PT(DamageObject<T>) slash_Ptr( new DamageObject<T>( DMG_SLASH_NAME, 5.0f, effect_info_collection ) );
        PT(DamageObject<T>) down_slash_Ptr( new MeleeDamageObject<T>( DMG_DOWN_SLASH_NAME, 5.0f, effect_info_collection,
                                                                 collision_segment_Ptr, ep_a, ep_v, distance*ep_p, distance, 0.09 ) );
        CombatCharacter<T>::add_damage_object( down_slash_Ptr );
        
        PT(CollisionHandlerEventNode<T>) down_slash_collider_Ptr( new CollisionHandlerEventNode<T>( DMG_DOWN_SLASH_NAME ) );
        down_slash_Ptr->DamageObject<T>::add_collider( down_slash_collider_Ptr );
        down_slash_collider_Ptr->collision_node()->add_solid( collision_segment_Ptr );
        
        down_slash_collider_Ptr->show();
        down_slash_collider_Ptr->reparent_to( *down_slash_Ptr );
        
        //down_slash_collider_Ptr->turn_on_from_collision();
        //down_slash_collider_Ptr->detach_node();
        
        // Event Handling
        down_slash_collider_Ptr->add_in_pattern();
        down_slash_collider_Ptr->add_event();
    }
 }
 
 /** FRIEND FUNCTIONS **/
 
 #undef MODEL_DIRECTORY
 #undef WALK_ANIMATION_DIRECTORY
 #undef WALK_ANIMATION_NAME
 #undef RUN_ANIMATION_DIRECTORY
 #undef RUN_ANIMATION_NAME
 //#undef DMG_SLASH_NAME
 #undef DMG_SIDE_SLASH_NAME
 #undef DMG_DOWN_SLASH_NAME
 
 #undef MODEL_SCALE
 #undef WALKING_SPEED
 #undef RUNNING_SPEED
 
 #endif // RALPH_H
 